<?php
use Neos\Flow\Tests\FunctionalTestCase;
use Newland\PageFrameProvider\Definition\RoutingConfigurationHelper;
use Newland\PageFrameProvider\Service\PageFrameContextService;

class RoutingConfigurationHelperTest extends FunctionalTestCase
{

    /** @var RoutingConfigurationHelper */
    protected $subject;

    /** @var PageFrameContextService */
    protected $context;

    public function setUp()
    {
        parent::setUp();

        $this->context = new PageFrameContextService();
        $this->context->setDimensions([ 'language' => [ 'de' ] ]);

        $this->subject = new RoutingConfigurationHelper();
        $this->inject($this->subject, 'pageFrameContext', $this->context);
        $this->subject->setTypeConfiguration(
            [
                'test-definition' => [
                    'uriSegment' => [
                        'de' => 'german/uri',
                        'en' => '/with/leading/and/trailing/slashes/',
                        '_fallback' => 'fallback/uri',
                    ],
                ],
                'test-definition-without-fallback' => [
                    'uriSegment' => [
                        'de' => 'german/uri',
                    ],
                ],
            ]
        );
    }

    public function testExtractsUriSegmentForType()
    {
        $this->context->setDimensions([ 'language' => [ 'de' ] ]);

        $this->assertEquals(
            'german/uri',
            $this->subject->urlFragmentForType('test-definition')
        );
    }

    public function testReturnsNullIfTypeUnknown()
    {
        $this->context->setDimensions([ 'language' => [ 'de' ] ]);

        $this->assertEquals(
            null,
            $this->subject->urlFragmentForType('unknown-type')
        );
    }

    public function testUsesFallbackDefinitionIfLanguageUnknown()
    {
        $this->context->setDimensions([ 'language' => [ 'ru' ] ]);

        $this->assertEquals(
            'fallback/uri',
            $this->subject->urlFragmentForType('test-definition')
        );
    }

    public function testReturnsNullIfTypeDefinedButLanguageUnknownAndNoFallbackLanguageDefined()
    {
        $this->context->setDimensions([ 'language' => [ 'ru' ] ]);

        $this->assertEquals(
            null,
            $this->subject->urlFragmentForType('test-definition-without-fallback')
        );
    }

    public function testStripsLeadingAndTrailingSlashesFromUri()
    {
        $this->context->setDimensions([ 'language' => [ 'en' ] ]);

        $this->assertEquals(
            'with/leading/and/trailing/slashes',
            $this->subject->urlFragmentForType('test-definition')
        );
    }

    public function testRemovesUriSegmentsFromRoutePath()
    {
        $this->context->setDimensions([ 'language' => [ 'de' ] ]);

        $this->assertEquals(
            '/foobar',
            $this->subject->removeUriSegmentsFromRoutePath('german/uri/foobar')
        );
    }

    public function testRemovesUriSegmentsFromRoutePathsThatAreDefinedWithLeadingAndTrailingSlashes()
    {
        $this->context->setDimensions([ 'language' => [ 'en' ] ]);

        $this->assertEquals(
            '/foobar',
            $this->subject->removeUriSegmentsFromRoutePath('with/leading/and/trailing/slashes/foobar')
        );
    }

    /**
     * Returns a route values array that is valid for the detail view if you add `--plugin._type`.
     *
     * @param array $overrides
     * @return array
     */
    private function routeValues(array $overrides = []): array
    {
        return array_replace_recursive(
            [
                '@package' => 'newland.pageframeprovider',
                '@controller' => 'pageframe',
                '@action' => 'show',
                '--plugin' => [
                    '@package' => 'newland.toubiz.poi.neos',
                    '@controller' => 'pointofinterests',
                    '@action' => 'show',
                    '_type' => '0',
                ],
            ],
            $overrides
        );
    }
}