<?php declare(strict_types=1);


namespace Newland\PageFrameProvider\Dimensions;

use Neos\Flow\Configuration\Exception\InvalidConfigurationException;
use Neos\Flow\ObjectManagement\Exception\InvalidObjectException;
use Neos\Flow\ObjectManagement\ObjectManager;
use Neos\Fusion\Exception as FusionException;
use Neos\Fusion\FusionObjects\JoinImplementation;
use Neos\Neos\Fusion\AbstractMenuItemsImplementation;
use Neos\Flow\Annotations as Flow;
use Newland\PageFrameProvider\Service\PageFrameContextService;
use Neos\Neos\Fusion\DimensionsMenuItemsImplementation as NeosDimensionsMenuImplementation;

class DimensionsMenuItemsImplementation extends NeosDimensionsMenuImplementation
{

    /**
     * @var PageFrameContextService
     * @Flow\Inject()
     */
    protected $pageFrameContext;

    /**
     * @var ObjectManager
     * @Flow\Inject
     */
    protected $objectManager;

    /**
     * @var array
     * @Flow\InjectConfiguration(package="Neos.ContentRepository", path="contentDimensions")
     */
    protected $dimensionConfiguration;


    /**
     * @inheritDoc
     */
    protected function buildItems()
    {
        $dimension = $this->getDimension();
        $dimensionConfig = $this->dimensionConfiguration[$dimension] ?? [];
        if (empty($dimensionConfig)) {
            return [];
        }

        $node = $this->pageFrameContext->getNode();
        $currentValue = $node->getDimensions()[$dimension][0] ?? null;

        $urls = $this->provider()->getAlternativeDimensions($dimension);
        $alternativeDimensions = [];
        foreach ($urls as $dimensionValue => $url) {
            $alternativeDimensions[] = [
                'label' => $this->getLabel($dimensionConfig, $dimensionValue) ?? $dimensionValue,
                'url' => $url,
                'value' => $dimensionValue,
                'state' => $dimensionValue === $currentValue
                    ? AbstractMenuItemsImplementation::STATE_CURRENT
                    : AbstractMenuItemsImplementation::STATE_NORMAL,
                'dimensions' => [ $dimension => [ $dimensionValue ] ],
            ];
        }

        return $alternativeDimensions;
    }


    protected function getLabel(array $dimensionConfig, string $dimensionValue): ?string
    {
        foreach ($dimensionConfig['presets'] ?? [] as $preset) {
            if (\in_array($dimensionValue, $preset['values'] ?? [])) {
                return $preset['label'];
            }
        }
        return null;
    }

    protected function provider(): AlternativeDimensionProvider
    {
        $config = $this->pageFrameContext->getAlternativeDimensionsConfig() ?: [];

        $className = NullProvider::class;
        if (array_key_exists('provider', $config)) {
            $className = $config['provider'];
        }

        if (!class_exists($className)) {
            throw new InvalidConfigurationException('No alternative language provider ' . $className . ' exists');
        }

        $instance = $this->objectManager->get($className);
        if (!($instance instanceof AlternativeDimensionProvider)) {
            throw new InvalidObjectException(
                'Alternative language providers must implement ' . AlternativeDimensionProvider::class
                . ' but ' . get_class($instance) . ' does not.'
            );
        }

        return $instance;
    }
}
