<?php
namespace Newland\NeosViewHelpers\ViewHelpers\Uri;

use Neos\Flow\Package\FlowPackageInterface;
use Neos\Flow\Package\PackageInterface;
use Neos\Flow\Package\PackageManager;
use Neos\FluidAdaptor\Core\Rendering\FlowAwareRenderingContextInterface;
use Neos\FluidAdaptor\ViewHelpers\Uri\ResourceViewHelper as OriginalResourceViewHelper;
use Neos\Utility\Files;
use TYPO3Fluid\Fluid\Core\Rendering\RenderingContextInterface;
use function Safe\md5_file;

/**
* uri.resource ViewHelper
* This ViewHelper can be used to render the uri to a resource including a hash of the file contents.
* This is good for browser cache busting - if the file changes, the hash changes and forces a fresh download.
*
* @example
* <code>
*   {namespace nv=Newland\NeosViewHelpers\ViewHelpers}
*   <script src="{nv:uri.resource(path: 'javascripts/main.js', package: 'Vendor.MyTheme')}" defer></script>
* </code>
* <output>
*   <script src="http://my-project.tld/_Resources/Static/Packages/Vendor.MyTheme/javascripts/
*     main.js?hash=0800fc577294c34e0b28ad2839435945" defer></script>
* </output>
 *
 * @example addHashParameter: false behaves similar to f:uri.resource
 * <code>
 *   {namespace nv=Newland\NeosViewHelpers\ViewHelpers}
 *   <script src="{nv:uri.resource(path: 'javascripts/main.js', package: 'Vendor.MyTheme', addHashParameter: false)}"
 *     defer></script>
 * </code>
 * <output>
 *   <script src="http://my-project.tld/_Resources/Static/Packages/Vendor.MyTheme/javascripts/main.js" defer></script>
 * </output>
 */
class ResourceViewHelper extends OriginalResourceViewHelper
{

    public static function renderStatic(
        array $arguments,
        \Closure $renderChildrenClosure,
        RenderingContextInterface $renderingContext
    ) {
        $url = parent::renderStatic($arguments, $renderChildrenClosure, $renderingContext);

        if ($arguments['addHashParameter'] === true) {
            $hash = self::getHash($arguments, $renderingContext);
            $url = self::addHash($url, $hash);
        }

        return $url;
    }

    private static function getPackage(array $arguments, RenderingContextInterface $renderingContext)
    {
        $package = null;

        if ($renderingContext instanceof FlowAwareRenderingContextInterface) {
            /** @var PackageManager $packageManager */
            $packageManager = $renderingContext->getObjectManager()->get(PackageManager::class);
            $package = $packageManager->getPackage($arguments['package']);
        }

        return $package;
    }

    private static function getHash(array $arguments, RenderingContextInterface $renderingContext): string
    {
        $package = self::getPackage($arguments, $renderingContext);

        return self::generateFileHash($arguments['path'], $package);
    }

    private static function addHash(string $url, string $hash): string
    {
        if ($hash !== '') {
            $url = $url . '?hash=' . $hash;
        }

        return $url;
    }

    private static function generateFileHash(string $path, PackageInterface $package = null): string
    {
        $hash = '';

        if ($package instanceof FlowPackageInterface) {
            $absolutePath = Files::concatenatePaths([ $package->getResourcesPath(), 'Public', $path ]);
            if (!file_exists($absolutePath)) {
                throw new \InvalidArgumentException(sprintf(
                    'File %s does not exist.',
                    $absolutePath
                ));
            }
            $hash = md5_file($absolutePath);
        }

        return $hash;
    }

    public function initializeArguments()
    {
        $this->registerArgument(
            'addHashParameter',
            'boolean',
            'Include a hash of the file contents as a query parameter',
            false,
            true
        );
        parent::initializeArguments();
    }

    public function render(): string
    {
        return self::renderStatic($this->arguments, $this->buildRenderChildrenClosure(), $this->renderingContext);
    }
}
