<?php declare(strict_types=1);
namespace Newland\NeosFiltering\Traits;

/*
 * This file is part of the "neos-filtering" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Contracts\Neos\Filter\DataSourcedFilterItem;
use Newland\Contracts\Neos\Filter\QueryBoundFilterItem;

/**
 * Default implementation for the `QueryBoundFilterItem` interface.
 * Note: This interface only provides a common implementation - you must explicitly implement
 *       the `QueryBoundFilterItem` interface if you want to mark your filter item as having a query string.
 *
 * @see QueryBoundFilterItem
 */
trait HasQueryString
{
    /** @var string */
    protected $queryString;

    public function getQueryString(): ?string
    {
        return $this->queryString;
    }

    public function setQueryString(string $queryString): void
    {
        $this->viewAssignments['queryString'] = $queryString;
        $this->queryString = $queryString;
    }

    protected function throwIfNoQueryString(): void
    {
        if ($this->queryString === null) {
            throw new \Exception('Please specify a query string');
        }
    }

    public function getQueryForOverrides(array $overrides): array
    {
        return $this->getQueryForOverridesFromDataSourcedFilterItem($overrides)
            ?? $this->getQueryForOverridesBySimpleStringMapping($overrides)
            ?? [];
    }

    private function getQueryForOverridesBySimpleStringMapping(array $overrides): ?array
    {
        $overrideProperty = $this->configuration['override']['property'] ?? null;
        $queryValue = $overrides[$overrideProperty] ?? null;

        if (!$queryValue) {
            return null;
        }

        return [ $this->queryString => $queryValue ];
    }

    private function getQueryForOverridesFromDataSourcedFilterItem(array $overrides): ?array
    {
        $query = [];
        $data = $this->dataFromDataSourceMappedToQueryString();
        if ($data === null) {
            return null;
        }

        foreach ($overrides as $property => $overrideValues) {
            if (($this->configuration['override']['property'] ?? null) === $property) {
                foreach ($overrideValues as $key => $overrideValue) {
                    $valueString = $data[$overrideValue] ?? null;
                    if ($valueString !== null) {
                        $queryString = $this->getQueryString();
                        $query[$queryString] = (array) ($query[$queryString] ?? []);
                        $query[$queryString][] = $valueString;
                    }
                }
            }
        }

        return $query;
    }

    private function dataFromDataSourceMappedToQueryString(): ?array
    {
        if (!($this instanceof DataSourcedFilterItem)) {
            return null;
        }

        $map = [];
        foreach ($this->getDataFromDataSource() as $data) {
            if (array_key_exists('queryString', $data)) {
                $map[$data['value']] = $data['queryString'];
            } else {
                $map[$data['value']] = $data['value'];
            }
        }
        return $map;
    }
}
