<?php
namespace Neos\Form\Tests\Functional;

/*
 * This file is part of the Neos.Form package.
 *
 * (c) Contributors of the Neos Project - www.neos.io
 *
 * This package is Open Source Software. For the full copyright and license
 * information, please view the LICENSE file which was distributed with this
 * source code.
 */
use Neos\Utility\ObjectAccess;
use Symfony\Component\DomCrawler\Field\InputFormField;

/**
 * Testcase for Simple Form
 *
 * @group large
 */
class SimpleFormTest extends AbstractFunctionalTestCase
{
    /**
     * @test
     */
    public function goingForthAndBackStoresFormValuesOfFirstPage()
    {
        $this->browser->request('http://localhost/test/form/simpleform/ThreePageFormWithValidation');

        $form = $this->browser->getForm();
        $form['--three-page-form-with-validation']['text1-1']->setValue('My Text on the first page');

        $this->gotoNextFormPage($form);

        $this->gotoPreviousFormPage($this->browser->getForm());

        $form = $this->browser->getForm();
        $this->assertSame('My Text on the first page', $form['--three-page-form-with-validation']['text1-1']->getValue());
    }

    /**
     * @test
     */
    public function goingForthAndBackStoresFormValuesOfSecondPage()
    {
        $this->browser->request('http://localhost/test/form/simpleform/ThreePageFormWithValidation');

        $this->gotoNextFormPage($this->browser->getForm());

        $form = $this->browser->getForm();
        $form['--three-page-form-with-validation']['text2-1']->setValue('My Text on the second page');
        $this->gotoPreviousFormPage($form);
        $this->gotoNextFormPage($this->browser->getForm());

        $form = $this->browser->getForm();
        $this->assertSame('My Text on the second page', $form['--three-page-form-with-validation']['text2-1']->getValue());
    }

    /**
     * @test
     */
    public function goingForthAndBackStoresFormValuesOfSecondPageAndTriggersValidationOnlyWhenGoingForward()
    {
        $this->browser->request('http://localhost/test/form/simpleform/ThreePageFormWithValidation');

        $this->gotoNextFormPage($this->browser->getForm());

        $form = $this->browser->getForm();
        $form['--three-page-form-with-validation']['text2-1']->setValue('My Text on the second page');
        $this->gotoPreviousFormPage($form);
        $this->gotoNextFormPage($this->browser->getForm());
        $this->gotoNextFormPage($this->browser->getForm());

        $this->assertSame(' error', $this->browser->getCrawler()->filterXPath('//*[contains(@class,"error")]//input[@id="three-page-form-with-validation-text2-1"]')->attr('class'));
        $form = $this->browser->getForm();
        $form['--three-page-form-with-validation']['text2-1']->setValue('42');
        $this->gotoNextFormPage($form);

        $form = $this->browser->getForm();
        $this->assertSame('', $form['--three-page-form-with-validation']['text3-1']->getValue());
    }

    /**
     * @test
     * Thanks to Anian Weber for reporting that issue!
     */
    public function validationIsNotSkippedForGetRequests()
    {
        $this->browser->request('http://localhost/test/form/simpleform/ThreePageFormWithValidation');

        // Navigate to 2nd form page
        $this->gotoNextFormPage($this->browser->getForm());

        $form = $this->browser->getForm();
        // Change form method to "GET"
        ObjectAccess::setProperty($form, 'method', 'GET', true);

        // Set invalid value (field "text2-1" has an IntegerValidator assigned)
        $form['--three-page-form-with-validation']['text2-1']->setValue('My Text on the second page');

        // Submit form
        $this->gotoNextFormPage($form);

        // Expect validation errors
        $this->assertSame(' error', $this->browser->getCrawler()->filterXPath('//*[contains(@class,"error")]//input[@id="three-page-form-with-validation-text2-1"]')->attr('class'));
    }

    /**
     * This is an edge-case which occurs if somebody makes the formState persistent, which can happen when subclassing the FormRuntime.
     *
     * The goal is to build a GET request *only* containing the form state, and nothing else. Furthermore, we need to make sure
     * that we do NOT send any of the parameters with the form; as we only want the form state to be applied.
     *
     * So, if the form state contains some values, we want to be sure these values are re-displayed.
     *
     * @test
     */
    public function goingForthAndBackStoresFormValuesOfSecondPageEvenWhenSecondPageIsManuallyCalledAsGetRequest()
    {
        $this->markTestSkipped('This test is skipped because we no longer allow Form validators to be skipped, see https://github.com/neos/form/security/advisories/GHSA-m5vx-8chx-qvmm');
        // 1. TEST SETUP: FORM STATE PREPARATION
        // - go to the 2nd page of the form, and fill in text2-1.
        $this->browser->request('http://localhost/test/form/simpleform/ThreePageFormWithValidation');

        $this->gotoNextFormPage($this->browser->getForm());

        $form = $this->browser->getForm();
        $form['--three-page-form-with-validation']['text2-1']->setValue('My Text on the second page');

        // - then, go back and forth, in order to get an *up-to-date* form state having the right values inside.
        $this->gotoPreviousFormPage($form);
        $this->gotoNextFormPage($this->browser->getForm());

        // 2. TEST SETUP: BUILD GET REQUEST ONLY CONTAINING FORM STATE
        $form = $this->browser->getForm();
        ObjectAccess::setProperty($form, 'method', 'GET', true);

        // we want to stay on the current page, that's why we send __currentPage = 1. (== 2nd page of the form)
        $doc = new \DOMDocument();
        $doc->loadXML('<input type="hidden" name="--three-page-form-with-validation[__currentPage]" value="1" />');
        $node = $doc->getElementsByTagName('input')->item(0);
        $form->set(new InputFormField($node));

        // We do *not* send any form content with us, as we want to test these are properly reconstituted from the form state.
        $form->offsetUnset('--three-page-form-with-validation[text2-1]');

        // 3. TEST RUN
        // submit the GET request ONLY containing formState.
        $this->browser->submit($form);

        // now, make sure the text2-1 (which has been persisted in the form state) gets reconstituted and shown properly.
        $form = $this->browser->getForm();
        $this->assertSame('My Text on the second page', $form['--three-page-form-with-validation']['text2-1']->getValue());
    }
}
