<?php
declare(strict_types=1);

namespace Neos\ContentRepository\Domain\Projection\Content;

/*
 * This file is part of the Neos.ContentRepository package.
 *
 * (c) Contributors of the Neos Project - www.neos.io
 *
 * This package is Open Source Software. For the full copyright and license
 * information, please view the LICENSE file which was distributed with this
 * source code.
 */

use Neos\ContentRepository\Domain\Model\NodeType;
use Neos\ContentRepository\Domain\ContentStream\ContentStreamIdentifier;
use Neos\ContentRepository\DimensionSpace\DimensionSpace\DimensionSpacePoint;
use Neos\ContentRepository\Domain\NodeAggregate\NodeAggregateIdentifier;
use Neos\ContentRepository\Domain\NodeAggregate\NodeName;
use Neos\ContentRepository\Domain\NodeType\NodeTypeName;
use Neos\Cache\CacheAwareInterface;

/**
 * This is a NEW interface, introduced in Neos 4.3.
 *
 * The new Event-Sourced core NodeInterface used for READING. It contains only information
 * local to a node; i.e. all properties in this interface can be accessed extremely fast.
 *
 * Most likely, you want to use {@see TraversableNodeInterface} instead, as it is more
 * convenient to use for end-users, as it also contains node traversal operations.
 *
 * The NodeInterface is *immutable*, meaning its contents never change after creation.
 * It is *only used for reading*.
 *
 * Starting with version 5.0 (when backed by the Event Sourced CR), it is
 * *completely detached from storage*; so it will not auto-update after a property changed in
 * storage.
 */
interface NodeInterface extends CacheAwareInterface
{
    /**
     * Whether or not this node is the root of the graph, i.e. has no parent node
     *
     * @return bool
     */
    public function isRoot(): bool;

    /**
     * Whether or not this node is tethered to its parent, fka auto created child node
     *
     * @return bool
     */
    public function isTethered(): bool;

    /**
     * @return ContentStreamIdentifier
     */
    public function getContentStreamIdentifier(): ContentStreamIdentifier;

    /**
     * @return NodeAggregateIdentifier
     */
    public function getNodeAggregateIdentifier(): NodeAggregateIdentifier;

    /**
     * @return NodeTypeName
     */
    public function getNodeTypeName(): NodeTypeName;

    /**
     * @return NodeType
     */
    public function getNodeType(): NodeType;

    /**
     * @return NodeName|null
     */
    public function getNodeName(): ?NodeName;

    /**
     * returns the DimensionSpacePoint the node is at home in. Usually needed to address a Node in a NodeAggregate
     * in order to update it.
     *
     * @return DimensionSpacePoint
     */
    public function getOriginDimensionSpacePoint(): DimensionSpacePoint;

    /**
     * Returns all properties of this node. References are NOT part of this API; there you need to check getReference() and getReferences()
     *
     * @return PropertyCollectionInterface Property values, indexed by their name
     * @api
     */
    public function getProperties(): PropertyCollectionInterface;

    /**
     * Returns the specified property.
     *
     * If the node has a content object attached, the property will be fetched
     * there if it is gettable.
     *
     * @param string $propertyName Name of the property
     * @return mixed value of the property
     * @api
     */
    public function getProperty($propertyName);

    /**
     * If this node has a property with the given name. Does NOT check the NodeType; but checks
     * for a non-NULL property value.
     *
     * @param string $propertyName
     * @return boolean
     * @api
     */
    public function hasProperty($propertyName): bool;

    /**
     * Returns the node label as generated by the configured node label generator
     *
     * @return string
     */
    public function getLabel(): string;
}
