import { GoogleMap } from './google-maps/google-map';
import { MapToolkit } from './maptoolkit/map-toolkit';
import { Leaflet } from './leaflet/leaflet';
import { stripMultilineIndention } from '../../common/string-utilities';
import { parseStringAsDom } from '@nimius/dom-utility';

const constructors = {
    'google-maps': GoogleMap,
    toursprung: MapToolkit,
    osm: Leaflet,
};


/**
 * @param {string} type
 * @param {HTMLElement} mapContainer
 * @param {AbstractMapWithMarkers~Options} options
 * @param {Observable<Marker[]>} markerProvider
 * @returns {AbstractMapWithMarkers}
 */
export function initializeMap(type, mapContainer, options) {
    for (const key of Object.keys(constructors)) {
        if (key === type) {
            return new constructors[type](mapContainer, options);
        }
    }

    console.warn(stripMultilineIndention(`
        mapType '${type}' unknown. Using OSM.
        Valid mapTypes are [${Object.keys(constructors).join(', ')}]
    `));
    return new Leaflet(mapContainer, options);
}

/**
 * @param {AbstractMapWithMarkers} map
 */
export function setupAutomaticClickHandling(map) {
    map.onMarkerClick(marker => {
        if (marker.tour && marker.tour.polygon && marker.tour.polygon.length > 0 && map.drawTourOntoMap) {
            map.drawTourOntoMap(marker, true);
        }
        if (marker.contentRenderer && map.openInfoWindow) {
            marker._markerDom = parseStringAsDom(marker.contentRenderer())[0];
            map.openInfoWindow(marker, marker._markerDom);

            const closeButtons = marker._markerDom
                .querySelectorAll('[data-toubiz-map-info-window-close-button]');
            for (const closeButton of closeButtons) {
                closeButton.addEventListener('click', () => map.closeInfoWindow());
            }
        }
    });
}
