import { Marker } from '../toubiz-map/data/marker';
import { getMapConstructors } from './static-map-pair-factory';

/**
 * @typedef {object} StaticMap~Marker
 * @property {number} latitude
 * @property {number} longitude
 * @property {string} tour
 * @property {string} iconUrl
 */

/**
 * @typedef {object} StaticMap~Options
 * @property {StaticMap~Marker[]} markers
 * @property {string} mapType
 * @property {AbstractMapWithMarkers~Options} [map = {}]
 */

/**
 * Static map: Displays a static map until the user clicks on it at which point a
 * dynamic map is being loaded.
 */
export class StaticMap {

    /**
     * @param {HTMLElement} node
     * @param {StaticMap~Options} settings
     */
    constructor (node, settings) {
        /** @private {HTMLElement} */
        this.node = node;

        /** @private {StaticMap~Options} */
        this.settings = settings;

        const markers = this.getMarkers();
        const image = this.initializeStaticMap(markers);
        image.addEventListener('click', async () => {
            this.setNodeSizeToImageSize(image);
            await this.initializeDynamicMap(markers);
            if (image.parentNode) {
                image.parentNode.removeChild(image);
            }
        });
    }

    /**
     * @private
     * @param {Marker[]} markers
     * @returns {HTMLImageElement}
     */
    initializeStaticMap(markers) {
        const image = document.createElement('img');
        this.node.appendChild(image);

        const config = this.settings.map || {};
        config.width = this.settings.width;
        config.height = this.settings.height;

        const maps = getMapConstructors(this.settings.mapType);
        const map = new maps.Static(image, config);
        map.render(markers);

        return image;
    }

    /**
     * @private
     * @param {Marker[]} markers
     * @returns {Promise<void>}
     */
    async initializeDynamicMap(markers) {
        const maps = getMapConstructors(this.settings.mapType);
        const map = new maps.Dynamic(this.node, this.settings.map || {});
        await map.render(markers);
        map.centerMapOnMarkers(markers);

        if (map.drawTourOntoMap) {
            markers.forEach(marker => {
                if (marker.tour && marker.tour.polygon) {
                    map.drawTourOntoMap(marker);
                }
            });
        }
    }

    /**
     * @private
     * @returns {Marker[]}
     */
    getMarkers() {
        return this.settings.markers.map(data => {
            const marker = new Marker();
            marker.coordinates = {
                latitude: parseFloat(data.latitude),
                longitude: parseFloat(data.longitude),
            };
            if (data.tour) {
                marker.tour = { polygon: JSON.parse(data.tour) };
            }
            if (data.iconUrl) {
                marker.style = {
                    height: 50,
                    width: 35,
                    offsetX: 25,
                    offsetY: 35,
                    iconUrl: data.iconUrl,
                };
            }
            return marker;
        });
    }

    setNodeSizeToImageSize(image) {
        const rect = image.getBoundingClientRect();
        this.node.style.width = `${rect.width}px`;
        this.node.style.height = `${rect.height}px`;
    }

}
